using System;
using System.Collections;
using System.Data;
using System.Text.RegularExpressions;
using System.Windows.Forms;
using System.Drawing;
using CONTROLS = gov.va.med.vbecs.GUI.controls;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.GUI
{
    /// <summary>
    /// Summary description for FrmAboRHConfirmProcess.
    /// </summary>
    public class FrmAboRHConfirmProcess : gov.va.med.vbecs.GUI.VbecsBaseForm
    {
        private System.ComponentModel.IContainer components;

		#region Variables 

		//Flag to indicate that code is "manually" advancing the cursor position -- to avoid extra movement
		private bool _manualAdvance = false;

		private int _previousRow = -1;		//Using this to refresh previous rows indicator

		//These constants will allow the code to distinguish "hidden" columns from visible ones
		private const int TOOLTIP_COLUMN = 14;
		private const int RIGHTMOST_COLUMN = 10;
		private const int LEFTMOST_COLUMN = 2;
		private const bool PROCESS_ACROSS = true;		//Set this if the user's decide to change their minds on navigation
		private const string INDICATOR_MSG = ARTIFICIAL.IndicatorMessage;		//Column name for where the IndicatorMessage goes

		//Flag to indicate that a column should be bypassed -- when we wrap, we need to skip the first two columns, as well as disabled columns
		private bool _byPassColumn = false;

		//DataTable that will correspond to each entry on the datagrid
		private System.Data.DataTable _dtBloodUnitTests;

		private int _EndEditColumn = 8;			//The last column that requires "Key Legend" -- aka Start of Interp Columns

		private System.Data.DataTable _dtABOConfirmationLookup;
		private System.Data.DataTable _dtRHConfirmationLookup;

		//Values saved off from FrmAboRhSelectInvoice -- to be passed in with Save
		private DateTime _testDateTime;
		private string _automatedInstrName;
		private bool _automatedInstrQCd;
		private Guid _rackGuid;
		private string _testUser;
		private bool _performAntiD;
		private string _rackNotQcComment = string.Empty;

		//Holds the previous Value of a cell -- used to restore prior value when override pop up and user cancels
		private string _prevCellValue = string.Empty;
		//Stores our current column -- used in navigation
		private int _currentColumn = int.MinValue;

		#endregion

		#region Form Controls

        private gov.va.med.vbecs.GUI.controls.ComboBoxLookup cmbABOComment;
        private gov.va.med.vbecs.GUI.controls.ComboBoxLookup cmbRhComment;

        private System.Windows.Forms.ToolTip toolTip1;
        private System.Windows.Forms.Panel pnlConfirm;
        private System.Windows.Forms.GroupBox grpWorklist;
        private System.Windows.Forms.GroupBox groupBox1;
        private System.Windows.Forms.Label lblValidEntries;
        private System.Windows.Forms.Panel pnlWorklist;
        private gov.va.med.vbecs.GUI.controls.WorklistBuilder worklistBuilder1;
        private gov.va.med.vbecs.GUI.controls.VbecsButton btnOK;
        private System.Windows.Forms.Button btnCancel;
        private System.Windows.Forms.GroupBox grpKey;
        private System.Windows.Forms.Label label1;
        private System.Windows.Forms.Label label2;
        private System.Windows.Forms.Label label3;
        private System.Windows.Forms.Label label4;
        private System.Windows.Forms.Label label5;
        private System.Windows.Forms.Label label6;
        private CONTROLS.VbecsDataGrid dgBloodUnitTests;

		//Grid layout style -- column names/widths etc...
		private System.Windows.Forms.DataGridTableStyle gridTableStyle;
		private System.Windows.Forms.DataGridTextBoxColumn gridTextBoxCol;
		private CONTROLS.DataGridKeyTrapTextBoxColumn gridKeyTrapTextBoxCol;
		private bool _keyPressEventHandled;

		#endregion

		#region Constructors/Destructors

        ///<Developers>
        ///	<Developer>Mike Smith</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/13/2002</CreationDate>
        ///
        /// <summary>
        /// Constructor to initialize the form.
        /// </summary>
        public FrmAboRHConfirmProcess(DateTime testDateTime, string testUser, string automatedInstrName, bool automatedInstrQCd, Guid rackGuid, bool performAntiD, string rackNotQcComment)
        {
            InitializeComponent();

            this.SetComponentProperties();

			this._testDateTime = testDateTime;
			this._testUser = testUser;
			this._automatedInstrName = automatedInstrName;
			this._automatedInstrQCd = automatedInstrQCd;
			this._rackGuid = rackGuid;
			this._performAntiD = performAntiD;
			this._rackNotQcComment = rackNotQcComment;

			this.SetThisHelpFile(this, Common.StrRes.HelpFiles.UC002.FrmAboRHConfirmProcess_Units());
        }

        ///<Developers>
        ///	<Developer>Mike Smith</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/13/2002</CreationDate>
        ///
        /// <summary>
        /// Constructor to initialize the form and create a list of all limited blood units for a 
        /// specific shipment.
        /// </summary>
		/// <param name="shipmentGuid"></param>
		/// <param name="testDateTime"></param>
		/// <param name="testUser"></param>
		/// <param name="automatedInstrName"></param>
		/// <param name="automatedInstrQCd"></param>
		/// <param name="rackGuid"></param>
		/// <param name="performAntiD"></param>
		/// <param name="rackNotQcComment"></param>
        public FrmAboRHConfirmProcess(System.Guid shipmentGuid, DateTime testDateTime, string testUser, string automatedInstrName, bool automatedInstrQCd, Guid rackGuid, bool performAntiD, string rackNotQcComment)
        {
			InitializeComponent();

			this.SetComponentProperties();
			
			this.worklistBuilder1.DTBloodUnits = BOL.BloodUnit.GetLimitedBloodUnitsByShipment(shipmentGuid, performAntiD);

			this._testDateTime = testDateTime;
			this._testUser = testUser;
			this._automatedInstrName = automatedInstrName;
			this._automatedInstrQCd = automatedInstrQCd;
			this._rackGuid = rackGuid;
			this._performAntiD = performAntiD;
			this._rackNotQcComment = rackNotQcComment;

			this.SetThisHelpFile(this, Common.StrRes.HelpFiles.UC002.FrmAboRHConfirmProcess());
        }

        /// <summary>
        /// Clean up any resources being used.
        /// </summary>
        protected override void Dispose( bool disposing )
        {
            if( disposing )
            {
                if(components != null)
                {
                    components.Dispose();
                }
            }
            base.Dispose( disposing );
        }
		#endregion

		#region Windows Form Designer generated code
        /// <summary>
        /// Required method for Designer support - do not modify
        /// the contents of this method with the code editor.
        /// </summary>
        private void InitializeComponent()
        {
			this.components = new System.ComponentModel.Container();
			this.toolTip1 = new System.Windows.Forms.ToolTip(this.components);
			this.pnlConfirm = new System.Windows.Forms.Panel();
			this.grpKey = new System.Windows.Forms.GroupBox();
			this.label6 = new System.Windows.Forms.Label();
			this.label5 = new System.Windows.Forms.Label();
			this.label4 = new System.Windows.Forms.Label();
			this.label3 = new System.Windows.Forms.Label();
			this.label2 = new System.Windows.Forms.Label();
			this.label1 = new System.Windows.Forms.Label();
			this.groupBox1 = new System.Windows.Forms.GroupBox();
			this.lblValidEntries = new System.Windows.Forms.Label();
			this.grpWorklist = new System.Windows.Forms.GroupBox();
			this.dgBloodUnitTests = new gov.va.med.vbecs.GUI.controls.VbecsDataGrid();
			this.cmbABOComment = new gov.va.med.vbecs.GUI.controls.ComboBoxLookup(this.components);
			this.cmbRhComment = new gov.va.med.vbecs.GUI.controls.ComboBoxLookup(this.components);
			this.pnlWorklist = new System.Windows.Forms.Panel();
			this.worklistBuilder1 = new gov.va.med.vbecs.GUI.controls.WorklistBuilder(this.components);
			this.btnOK = new gov.va.med.vbecs.GUI.controls.VbecsButton();
			this.btnCancel = new System.Windows.Forms.Button();
			this.pnlConfirm.SuspendLayout();
			this.grpKey.SuspendLayout();
			this.groupBox1.SuspendLayout();
			this.grpWorklist.SuspendLayout();
			((System.ComponentModel.ISupportInitialize)(this.dgBloodUnitTests)).BeginInit();
			this.pnlWorklist.SuspendLayout();
			this.SuspendLayout();
			// 
			// vbecsLogo1
			// 
			this.vbecsLogo1.Name = "vbecsLogo1";
			this.vbecsLogo1.Size = new System.Drawing.Size(824, 30);
			this.vbecsLogo1.TabIndex = 4;
			this.vbecsLogo1.Title = "Unit ABO/Rh Confirmation";
			// 
			// lblRequiredField
			// 
			this.lblRequiredField.Location = new System.Drawing.Point(0, 469);
			this.lblRequiredField.Name = "lblRequiredField";
			this.lblRequiredField.TabIndex = 5;
		
			// toolTip1
			// 
			this.toolTip1.AutomaticDelay = 250;
			this.toolTip1.AutoPopDelay = 5000;
			this.toolTip1.InitialDelay = 250;
			this.toolTip1.ReshowDelay = 50;
			// 
			// pnlConfirm
			// 
			this.pnlConfirm.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.pnlConfirm.Controls.Add(this.grpKey);
			this.pnlConfirm.Controls.Add(this.groupBox1);
			this.pnlConfirm.Controls.Add(this.grpWorklist);
			this.pnlConfirm.Location = new System.Drawing.Point(0, 32);
			this.pnlConfirm.Name = "pnlConfirm";
			this.pnlConfirm.Size = new System.Drawing.Size(819, 416);
			this.pnlConfirm.TabIndex = 1;
			this.pnlConfirm.Visible = false;
			this.pnlConfirm.VisibleChanged += new System.EventHandler(this.pnlConfirm_VisibleChanged);
			// 
			// grpKey
			// 
			this.grpKey.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Left)));
			this.grpKey.Controls.Add(this.label6);
			this.grpKey.Controls.Add(this.label5);
			this.grpKey.Controls.Add(this.label4);
			this.grpKey.Controls.Add(this.label3);
			this.grpKey.Controls.Add(this.label2);
			this.grpKey.Controls.Add(this.label1);
			this.grpKey.Location = new System.Drawing.Point(4, 342);
			this.grpKey.Name = "grpKey";
			this.grpKey.Size = new System.Drawing.Size(100, 72);
			this.grpKey.TabIndex = 1;
			this.grpKey.TabStop = false;
			this.grpKey.Text = "Key";
			// 
			// label6
			// 
			this.label6.Location = new System.Drawing.Point(36, 48);
			this.label6.Name = "label6";
			this.label6.Size = new System.Drawing.Size(48, 16);
			this.label6.TabIndex = 5;
			this.label6.Text = "Error";
			this.label6.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// label5
			// 
			this.label5.BackColor = System.Drawing.Color.Red;
			this.label5.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
			this.label5.Location = new System.Drawing.Point(8, 48);
			this.label5.Name = "label5";
			this.label5.Size = new System.Drawing.Size(16, 16);
			this.label5.TabIndex = 4;
			this.label5.Text = "E";
			this.label5.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
			// 
			// label4
			// 
			this.label4.Location = new System.Drawing.Point(36, 32);
			this.label4.Name = "label4";
			this.label4.Size = new System.Drawing.Size(49, 16);
			this.label4.TabIndex = 3;
			this.label4.Text = "Warning";
			this.label4.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// label3
			// 
			this.label3.Location = new System.Drawing.Point(36, 16);
			this.label3.Name = "label3";
			this.label3.Size = new System.Drawing.Size(48, 16);
			this.label3.TabIndex = 1;
			this.label3.Text = "Valid";
			this.label3.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// label2
			// 
			this.label2.BackColor = System.Drawing.Color.Yellow;
			this.label2.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
			this.label2.Location = new System.Drawing.Point(8, 32);
			this.label2.Name = "label2";
			this.label2.Size = new System.Drawing.Size(16, 16);
			this.label2.TabIndex = 2;
			this.label2.Text = "W";
			this.label2.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
			// 
			// label1
			// 
			this.label1.BackColor = System.Drawing.Color.Lime;
			this.label1.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
			this.label1.CausesValidation = false;
			this.label1.Location = new System.Drawing.Point(8, 16);
			this.label1.Name = "label1";
			this.label1.Size = new System.Drawing.Size(16, 16);
			this.label1.TabIndex = 0;
			this.label1.Text = "V";
			this.label1.TextAlign = System.Drawing.ContentAlignment.MiddleCenter;
			// 
			// groupBox1
			// 
			this.groupBox1.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.groupBox1.Controls.Add(this.lblValidEntries);
			this.groupBox1.Location = new System.Drawing.Point(108, 342);
			this.groupBox1.Name = "groupBox1";
			this.groupBox1.Size = new System.Drawing.Size(708, 72);
			this.groupBox1.TabIndex = 2;
			this.groupBox1.TabStop = false;
			this.groupBox1.Text = "Valid Entries";
			// 
			// lblValidEntries
			// 
			this.lblValidEntries.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.lblValidEntries.Font = new System.Drawing.Font("Microsoft Sans Serif", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.lblValidEntries.Location = new System.Drawing.Point(8, 16);
			this.lblValidEntries.Name = "lblValidEntries";
			this.lblValidEntries.Size = new System.Drawing.Size(688, 48);
			this.lblValidEntries.TabIndex = 0;
			this.lblValidEntries.Text = "Valid Entries";
			// 
			// grpWorklist
			// 
			this.grpWorklist.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.grpWorklist.Controls.Add(this.dgBloodUnitTests);
			this.grpWorklist.Location = new System.Drawing.Point(4, 0);
			this.grpWorklist.Name = "grpWorklist";
			this.grpWorklist.Size = new System.Drawing.Size(816, 342);
			this.grpWorklist.TabIndex = 0;
			this.grpWorklist.TabStop = false;
			// 
			// dgBloodUnitTests
			// 
			this.dgBloodUnitTests.AllowSorting = false;
			this.dgBloodUnitTests.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.dgBloodUnitTests.CaptionFont = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Bold, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.dgBloodUnitTests.CaptionVisible = false;
			this.dgBloodUnitTests.DataMember = "";
			this.dgBloodUnitTests.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.dgBloodUnitTests.HeaderFont = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.dgBloodUnitTests.HeaderForeColor = System.Drawing.SystemColors.ControlText;
			this.dgBloodUnitTests.Location = new System.Drawing.Point(4, 12);
			this.dgBloodUnitTests.Name = "dgBloodUnitTests";
			this.dgBloodUnitTests.RowHeadersVisible = false;
			this.dgBloodUnitTests.Size = new System.Drawing.Size(812, 322);
			this.dgBloodUnitTests.TabIndex = 0;
			this.dgBloodUnitTests.ToolTipColumn = -1;
			this.dgBloodUnitTests.CurrentCellChanged += new System.EventHandler(this.dgBloodUnitTests_CurrentCellChanged);
			this.dgBloodUnitTests.SizeChanged += new System.EventHandler(this.dgBloodUnitTests_SizeChanged);
			// 
			// cmbABOComment
			// 
			this.cmbABOComment.Interval = 500;
			this.cmbABOComment.Location = new System.Drawing.Point(0, 0);
			this.cmbABOComment.MaxLength = 255;
			this.cmbABOComment.Name = "cmbABOComment";
			this.cmbABOComment.Size = new System.Drawing.Size(128, 21);
			this.cmbABOComment.TabIndex = 0;
			this.cmbABOComment.VisibleChanged += new System.EventHandler(this.cmbABOComment_VisibleChanged);
			this.cmbABOComment.KeyDown += new System.Windows.Forms.KeyEventHandler(this.cmbABOComment_KeyDown);
			this.cmbABOComment.TextChanged += new System.EventHandler(this.cmbABOComment_TextChanged);
			// 
			// cmbRhComment
			// 
			this.cmbRhComment.Interval = 500;
			this.cmbRhComment.Location = new System.Drawing.Point(0, 0);
			this.cmbRhComment.MaxLength = 255;
			this.cmbRhComment.Name = "cmbRhComment";
			this.cmbRhComment.Size = new System.Drawing.Size(128, 21);
			this.cmbRhComment.TabIndex = 0;
			this.cmbRhComment.VisibleChanged += new System.EventHandler(this.cmbRhComment_VisibleChanged);
			this.cmbRhComment.KeyDown += new System.Windows.Forms.KeyEventHandler(this.cmbRhComment_KeyDown);
			this.cmbRhComment.TextChanged += new System.EventHandler(this.cmbRhComment_TextChanged);
			// 
			// pnlWorklist
			// 
			this.pnlWorklist.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.pnlWorklist.Controls.Add(this.worklistBuilder1);
			this.pnlWorklist.Location = new System.Drawing.Point(0, 32);
			this.pnlWorklist.Name = "pnlWorklist";
			this.pnlWorklist.Size = new System.Drawing.Size(819, 416);
			this.pnlWorklist.TabIndex = 0;
			// 
			// worklistBuilder1
			// 
			this.worklistBuilder1.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.worklistBuilder1.CheckBoxes = true;
			this.worklistBuilder1.Location = new System.Drawing.Point(8, 8);
			this.worklistBuilder1.Name = "worklistBuilder1";
			this.worklistBuilder1.Size = new System.Drawing.Size(809, 408);
			this.worklistBuilder1.TabIndex = 0;
			this.worklistBuilder1.InvalidWorklist += new System.EventHandler(this.CheckValid);
			this.worklistBuilder1.ValidWorklist += new System.EventHandler(this.CheckValid);
			this.worklistBuilder1.SearchClick += new System.EventHandler(this.worklistBuilder1_SearchClick);
			this.worklistBuilder1.SelectBtnClick += new System.EventHandler(this.worklistBuilder1_SelectBtnClick);
			this.worklistBuilder1.CheckBoxesChanged += new System.EventHandler(this.CheckValid);
			this.worklistBuilder1.UnitsInList += new System.EventHandler(this.CheckValid);
			this.worklistBuilder1.ItemCheck += new System.EventHandler(this.worklistBuilder1_ItemCheck);
			// 
			// btnOK
			// 
			this.btnOK.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Right)));
			this.btnOK.DialogResult = System.Windows.Forms.DialogResult.OK;
			this.btnOK.EnabledButton = false;
			this.btnOK.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.btnOK.Image = null;
			this.btnOK.Location = new System.Drawing.Point(664, 456);
			this.btnOK.Name = "btnOK";
			this.btnOK.Size = new System.Drawing.Size(75, 23);
			this.btnOK.TabIndex = 2;
			this.btnOK.Click += new System.EventHandler(this.btnOK_Click);
			// 
			// btnCancel
			// 
			this.btnCancel.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Right)));
			this.btnCancel.DialogResult = System.Windows.Forms.DialogResult.Cancel;
			this.btnCancel.Location = new System.Drawing.Point(744, 456);
			this.btnCancel.Name = "btnCancel";
			this.btnCancel.TabIndex = 3;
			this.btnCancel.Text = "&Cancel";
			this.btnCancel.Click += new System.EventHandler(this.btnCancel_Click);
			// 
			// FrmAboRHConfirmProcess
			// 
			this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
			this.CancelButton = this.btnCancel;
			this.ClientSize = new System.Drawing.Size(824, 485);
			this.Controls.Add(this.btnCancel);
			this.Controls.Add(this.btnOK);
			this.Controls.Add(this.pnlConfirm);
			this.Controls.Add(this.pnlWorklist);
			this.hlphelpChm.SetHelpKeyword(this, "ABO_Rh_Confirmation.htm");
			this.hlphelpChm.SetHelpNavigator(this, System.Windows.Forms.HelpNavigator.Topic);
			this.MinimumSize = new System.Drawing.Size(830, 490);
			this.Name = "FrmAboRHConfirmProcess";
			this.RequiredFieldsLabel = true;
			this.hlphelpChm.SetShowHelp(this, true);
			this.Text = "VBECS - Unit ABO/Rh Confirmation";
			this.Load += new System.EventHandler(this.FrmAboRHConfirmProcess_Load);
			this.Controls.SetChildIndex(this.pnlWorklist, 0);
			this.Controls.SetChildIndex(this.pnlConfirm, 0);
			this.Controls.SetChildIndex(this.btnOK, 0);
			this.Controls.SetChildIndex(this.btnCancel, 0);
			this.Controls.SetChildIndex(this.vbecsLogo1, 0);
			this.Controls.SetChildIndex(this.lblRequiredField, 0);
			this.pnlConfirm.ResumeLayout(false);
			this.grpKey.ResumeLayout(false);
			this.groupBox1.ResumeLayout(false);
			this.grpWorklist.ResumeLayout(false);
			((System.ComponentModel.ISupportInitialize)(this.dgBloodUnitTests)).EndInit();
			this.pnlWorklist.ResumeLayout(false);
			this.ResumeLayout(false);

		}
		#endregion

		#region Methods

		/// <summary>
		/// Form Setup Calls and commands
		/// </summary>
		public void SetComponentProperties()
		{
            this.LockController.LockFunction = Common.LockFunctions.UC002AboRHConfirmation;

			this.vbecsLogo1.Title = this.Text.Replace("VBECS - ","");
			// CR 2693
			// hlpHelp
			// 
			this.hlpHelp.HelpNamespace = Environment.CurrentDirectory+"\\VBECS.chm";
			// 
			// hlphelpChm
			// 
			this.hlphelpChm.HelpNamespace = Environment.CurrentDirectory+"\\VBECS.chm";
			// 
			// 

			dgBloodUnitTests.ToolTipColumn = TOOLTIP_COLUMN;

			this.PopulateCannedCommentsList();

			//Validate that our Truth Table data is present -- otherwise the validations will all be errors ---
			this._dtABOConfirmationLookup = BOL.BloodUnitTest.GetAboConfirmationLookup();
			if (this._dtABOConfirmationLookup.Rows.Count == 0)
			{
				GuiMessenger.ShowMessageBox(Common.StrRes.SysErrMsg.Common.NoDataRowsFoundException(TABLE.ValidAboResult.TableName));
				this.DialogResult = DialogResult.Abort;
				this.Close();
				return;
			}

			//Validate that our Truth Table data is present -- otherwise the validations will all be errors ---
			this._dtRHConfirmationLookup = BOL.BloodUnitTest.GetRHConfirmationLookup();
			if (this._dtRHConfirmationLookup.Rows.Count == 0)
			{
				GuiMessenger.ShowMessageBox(Common.StrRes.SysErrMsg.Common.NoDataRowsFoundException(TABLE.ValidRhResult.TableName));
				this.DialogResult = DialogResult.Abort;
				this.Close();
				return;
			}

			this.btnOK.SetToolTip(Common.StrRes.Tooltips.UC002.SelectUnit().ResString);
			this.toolTip1.SetToolTip(this.btnCancel, Common.StrRes.Tooltips.Common.ClickToDisregardChangesAndReturn().ResString);

			//This allows us to set our own grid movement logic 
			this.dgBloodUnitTests.NextCell += new vbecs.GUI.controls.NextCellEventHandler(this.OnNextCellChanged);
		}



		/// <summary>
		/// Converts grid entry into TT2.01-2 relatable value (P,N,I,X) - Does not translate Interps!
		/// </summary>
		/// <param name="gridText"></param>
		/// <returns></returns>
		public string GetResultTranslation(string gridText)
		{
			switch (gridText)
			{
					//Not Tested - Null - Remove Null/Blank TT2.01 Step 7, NOT ALLOWED
					//@ = set in code via UpdateIndicator, means the column is diabled, treat as X
				case "X":  case "@":
					return "X";

					//Positive Values		//1-30-2005 - Lohse Removing "P" for CR332
				case "W":	case "F":	case "H":  case "M":	case "1":	case "2":	case "3":	case "4":
					return "P";	
				
					//Negative Values		//1-30-2005 - Lohse - Removing "N" for CR332
				case "0":
					return "N";
			
				default:		//Column is avaialbe for entry, but user hasn't entered anything yet
					return "~";	
			}
		}



		/// <summary>
		/// Fires when the user moves between cells
		/// </summary>
		private void OnNextCellChanged()
		{
			this._manualAdvance = true;	//Let other listening events know that we're doin this manually

			if (PROCESS_ACROSS)
			{
				if (this.dgBloodUnitTests.CurrentCell.ColumnNumber + 1 > RIGHTMOST_COLUMN)
				{
					this.cmbRhComment.Visible = false;
					if (this.dgBloodUnitTests.CurrentCell.RowNumber + 1 == this.dgBloodUnitTests.VisibleRowCount)
					{
						this.dgBloodUnitTests.CurrentCell = new DataGridCell(0,LEFTMOST_COLUMN);
					}
					else
					{
						this.dgBloodUnitTests.CurrentCell = new DataGridCell(this.dgBloodUnitTests.CurrentCell.RowNumber + 1, LEFTMOST_COLUMN);
					}
				}
				else
				{
					if (this.dgBloodUnitTests.CurrentCell.ColumnNumber + 1 == RIGHTMOST_COLUMN)
					{
						this.cmbABOComment.Visible = false;
						this.cmbRhComment.Visible = true;
						this.cmbRhComment.BringToFront();
						this.cmbRhComment.Show();
						this.cmbRhComment.Focus();
					}
					if (this.dgBloodUnitTests.CurrentCell.ColumnNumber + 2 == RIGHTMOST_COLUMN)
					{
						this.cmbABOComment.Visible = true;
						this.cmbABOComment.BringToFront();
						this.cmbABOComment.Show();
						this.cmbABOComment.Focus();
					}

					if (this._dtBloodUnitTests.Rows[this.dgBloodUnitTests.CurrentCell.RowNumber].IsNull(this.dgBloodUnitTests.CurrentCell.ColumnNumber + 1))
					{
						if (this.dgBloodUnitTests.CurrentCell.ColumnNumber + 3 == RIGHTMOST_COLUMN)  //If the next enabled is the CmbBox, we have to make it visible
						{
							this.cmbRhComment.Visible = true;
							this.cmbRhComment.BringToFront();
							this.cmbRhComment.Show();
							this.cmbRhComment.Focus();
						}
					}

					this._manualAdvance = false;
					this.dgBloodUnitTests.CurrentCell = new DataGridCell(this.dgBloodUnitTests.CurrentCell.RowNumber, this.dgBloodUnitTests.CurrentCell.ColumnNumber + 1);
				}
			}
			this._manualAdvance = false;
		}


        ///<Developers>
        ///	<Developer>Mike Smith</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/13/2002</CreationDate>
        ///
        /// <summary>
        /// Calls the method to create a hash table of testing results, creates the blood unit test table, 
        /// populates the table with the selected units, and sets the current cell to allow editing.
        /// </summary>
        private void InitializeGrid()
        {
            this._dtBloodUnitTests = BOL.BloodUnitTest.GetEmptyAboRhConfirmationTable(); 

            AddCustomDataTableStyle();

            PopulateBloodUnitTests();

            dgBloodUnitTests.DataSource = _dtBloodUnitTests.DefaultView;

			dgBloodUnitTests.RightMostColumn = RIGHTMOST_COLUMN + 1;
	
            ((DataView)dgBloodUnitTests.DataSource).AllowNew = false;
            ((DataView)dgBloodUnitTests.DataSource).AllowDelete = false;

            if (worklistBuilder1.SelectedBloodBags.Rows.Count > 0)
            {
                dgBloodUnitTests.CurrentCell = new DataGridCell(0,LEFTMOST_COLUMN);
            }
			
			//Fire off the size event so that the comment fields are formatted fully
			this.dgBloodUnitTests_SizeChanged(this, System.EventArgs.Empty);

			//Default the OK Button to disabled
			this.btnOK.SetToolTip(Common.StrRes.Tooltips.Common.OKToSave().ResString);
			this.btnOK.EnabledButton = false;
			this._keyPressEventHandled = false;
		}



		/// <summary>
		/// Populates the Abo/Rh Comment ComboBoxes used on the Grid with CannedCommments
		/// </summary>
        private void PopulateCannedCommentsList()
        {
            //Get the comment list of "Unit Testing" categorytype
            System.Data.DataTable dtComments = BOL.CannedComment.GetCannedComments(Common.LogonUser.LogonUserDivisionCode, "UT");

            this.cmbABOComment.Items.Clear();
            this.cmbRhComment.Items.Clear();
			
			//Users are allowed to enter up to 255 chars of free text
			this.cmbABOComment.AllowManualEntry = true;
			this.cmbRhComment.AllowManualEntry = true;

            //Add Canned Comments from database
            foreach (System.Data.DataRow dr in dtComments.Rows)
            {
                this.cmbABOComment.Items.Add(dr[TABLE.CannedComment.CannedCommentText].ToString());
				this.cmbRhComment.Items.Add(dr[TABLE.CannedComment.CannedCommentText].ToString());
            }

        }


        ///<Developers>
        ///	<Developer>Mike Smith</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/13/2002</CreationDate>
        ///
        /// <summary>
        /// Adds the custom data table style to the data grid.
        /// </summary>
        private void AddCustomDataTableStyle()
        {
            gridTableStyle = new DataGridTableStyle();
            //Defect 227411
            System.Drawing.Font myFont = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)), true);

            
            gridTableStyle.PreferredRowHeight = 22;

            gridTableStyle.AllowSorting = false;
            gridTableStyle.DataGrid = dgBloodUnitTests;

            gridTableStyle.HeaderFont = myFont;
            gridTableStyle.HeaderForeColor = System.Drawing.SystemColors.ControlText;
            gridTableStyle.MappingName = "";
            gridTableStyle.RowHeadersVisible = false;


            // 
            // Indicator
            // 
            gridKeyTrapTextBoxCol = new CONTROLS.DataGridKeyTrapTextBoxColumn();
            gridKeyTrapTextBoxCol.Alignment = HorizontalAlignment.Center;
            gridKeyTrapTextBoxCol.MappingName = ARTIFICIAL.Indicator;
            gridKeyTrapTextBoxCol.ReadOnly = true;
            gridKeyTrapTextBoxCol.Width = 20;
            gridTableStyle.GridColumnStyles.Add(gridKeyTrapTextBoxCol);

            // 
            // UnitID
            // 
            gridTextBoxCol = new DataGridTextBoxColumn();
            gridTextBoxCol.HeaderText = "Unit ID";
            gridTextBoxCol.MappingName = TABLE.BloodUnit.EyeReadableUnitId;
            gridTextBoxCol.ReadOnly = true;
            gridTextBoxCol.Width = 110;
            gridTableStyle.GridColumnStyles.Add(gridTextBoxCol);

            // 
            // AntiA
            // 
            gridKeyTrapTextBoxCol = new CONTROLS.DataGridKeyTrapTextBoxColumn();

            gridKeyTrapTextBoxCol.Alignment = HorizontalAlignment.Center;
            gridKeyTrapTextBoxCol.HeaderText = "  Anti A";
            gridKeyTrapTextBoxCol.MappingName = TABLE.ValidAboResult.AntiA;
            gridKeyTrapTextBoxCol.Width = 50;
            gridKeyTrapTextBoxCol.ProcessAcross = PROCESS_ACROSS;
            gridTableStyle.GridColumnStyles.Add(gridKeyTrapTextBoxCol);
            gridKeyTrapTextBoxCol.TextBox.KeyPress += new KeyPressEventHandler(gridKeyTrapTextBoxCol_TextBoxKeyPress);
            gridKeyTrapTextBoxCol.TextBox.TextChanged += new EventHandler(gridKeyTrapTextBoxCol_TextChanged);

            // 
            // AntiB
            // 
            gridKeyTrapTextBoxCol = new CONTROLS.DataGridKeyTrapTextBoxColumn();
            gridKeyTrapTextBoxCol.Alignment = HorizontalAlignment.Center;
            gridKeyTrapTextBoxCol.HeaderText = "  Anti B";
            gridKeyTrapTextBoxCol.MappingName = TABLE.ValidAboResult.AntiB;
            gridKeyTrapTextBoxCol.Width = 50;
            gridKeyTrapTextBoxCol.ProcessAcross = PROCESS_ACROSS;
            gridTableStyle.GridColumnStyles.Add(gridKeyTrapTextBoxCol);
            gridKeyTrapTextBoxCol.TextBox.KeyPress += new KeyPressEventHandler(gridKeyTrapTextBoxCol_TextBoxKeyPress);
            gridKeyTrapTextBoxCol.TextBox.TextChanged += new EventHandler(gridKeyTrapTextBoxCol_TextChanged);

            // 
            // AntiAB
            // 
            gridKeyTrapTextBoxCol = new CONTROLS.DataGridKeyTrapTextBoxColumn();
            gridKeyTrapTextBoxCol.Alignment = HorizontalAlignment.Center;
            gridKeyTrapTextBoxCol.HeaderText = " Anti A,B";
            gridKeyTrapTextBoxCol.MappingName = TABLE.ValidAboResult.AntiAB;
            gridKeyTrapTextBoxCol.Width = 50;
            gridKeyTrapTextBoxCol.ProcessAcross = PROCESS_ACROSS;
            gridTableStyle.GridColumnStyles.Add(gridKeyTrapTextBoxCol);
            gridKeyTrapTextBoxCol.TextBox.KeyPress += new KeyPressEventHandler(gridKeyTrapTextBoxCol_TextBoxKeyPress);
            gridKeyTrapTextBoxCol.TextBox.TextChanged += new EventHandler(gridKeyTrapTextBoxCol_TextChanged);

            // 
            // AntiD
            // 
            gridKeyTrapTextBoxCol = new CONTROLS.DataGridKeyTrapTextBoxColumn();
            gridKeyTrapTextBoxCol.Alignment = HorizontalAlignment.Center;
            gridKeyTrapTextBoxCol.HeaderText = "  Anti D";
            gridKeyTrapTextBoxCol.MappingName = TABLE.ValidRhResult.AntiD;
            gridKeyTrapTextBoxCol.Width = 50;
            gridKeyTrapTextBoxCol.ProcessAcross = PROCESS_ACROSS;
            gridTableStyle.GridColumnStyles.Add(gridKeyTrapTextBoxCol);
            gridKeyTrapTextBoxCol.TextBox.KeyPress += new KeyPressEventHandler(gridKeyTrapTextBoxCol_TextBoxKeyPress);
            gridKeyTrapTextBoxCol.TextBox.TextChanged += new EventHandler(gridKeyTrapTextBoxCol_TextChanged);

            // 
            // ControlD
            // 
            gridKeyTrapTextBoxCol = new CONTROLS.DataGridKeyTrapTextBoxColumn();
            gridKeyTrapTextBoxCol.Alignment = HorizontalAlignment.Center;
            gridKeyTrapTextBoxCol.HeaderText = " D Control";
            gridKeyTrapTextBoxCol.MappingName = TABLE.ValidRhResult.DControl;
            gridKeyTrapTextBoxCol.Width = 60;
            gridKeyTrapTextBoxCol.ProcessAcross = PROCESS_ACROSS;
            gridTableStyle.GridColumnStyles.Add(gridKeyTrapTextBoxCol);
            gridKeyTrapTextBoxCol.TextBox.KeyPress += new KeyPressEventHandler(gridKeyTrapTextBoxCol_TextBoxKeyPress);
            gridKeyTrapTextBoxCol.TextBox.TextChanged += new EventHandler(gridKeyTrapTextBoxCol_TextChanged);

            // 
            // AboInterp
            // 
            gridKeyTrapTextBoxCol = new CONTROLS.DataGridKeyTrapTextBoxColumn();
            gridKeyTrapTextBoxCol.Alignment = HorizontalAlignment.Center;
            gridKeyTrapTextBoxCol.HeaderText = " ABO Interp.";
            gridKeyTrapTextBoxCol.MappingName = TABLE.ValidAboResult.AboInterp;
            gridKeyTrapTextBoxCol.Width = 65;
            gridKeyTrapTextBoxCol.ProcessAcross = PROCESS_ACROSS;
            gridTableStyle.GridColumnStyles.Add(gridKeyTrapTextBoxCol);
            gridKeyTrapTextBoxCol.TextBox.KeyPress += new KeyPressEventHandler(gridKeyTrapTextBoxCol_TextBoxKeyPress);
            gridKeyTrapTextBoxCol.TextBox.TextChanged += new EventHandler(gridKeyTrapTextBoxCol_TextChanged);

            // 
            // RhInterp
            // 
            gridKeyTrapTextBoxCol = new CONTROLS.DataGridKeyTrapTextBoxColumn();
            gridKeyTrapTextBoxCol.Alignment = HorizontalAlignment.Center;
            gridKeyTrapTextBoxCol.HeaderText = " Rh Interp.";
            gridKeyTrapTextBoxCol.MappingName = TABLE.ValidRhResult.RhInterp;
            gridKeyTrapTextBoxCol.Width = 65;
            gridKeyTrapTextBoxCol.ProcessAcross = PROCESS_ACROSS;
            gridTableStyle.GridColumnStyles.Add(gridKeyTrapTextBoxCol);
            gridKeyTrapTextBoxCol.TextBox.KeyPress += new KeyPressEventHandler(gridKeyTrapTextBoxCol_TextBoxKeyPress);
            gridKeyTrapTextBoxCol.TextBox.TextChanged += new EventHandler(gridKeyTrapTextBoxCol_TextChanged);

            // 
            // AboComment
            // 
            gridTextBoxCol = new DataGridTextBoxColumn();
            gridTextBoxCol.HeaderText = "ABO Comment";
            gridTextBoxCol.MappingName = ARTIFICIAL.AboComment;
            gridTextBoxCol.Width = 130;
            gridTextBoxCol.TextBox.Controls.Add(cmbABOComment);
            cmbABOComment.BringToFront();
            gridTextBoxCol.ReadOnly = true;
            gridTableStyle.GridColumnStyles.Add(gridTextBoxCol);
            // 
            // RhComment
            // 
            gridTextBoxCol = new DataGridTextBoxColumn();
            gridTextBoxCol.HeaderText = "Rh Comment";
            gridTextBoxCol.MappingName = ARTIFICIAL.RhComment;
            gridTextBoxCol.Width = 130;
            gridTextBoxCol.ReadOnly = true;
            gridTextBoxCol.TextBox.Controls.Add(cmbRhComment);
            cmbRhComment.BringToFront();
            gridTableStyle.GridColumnStyles.Add(gridTextBoxCol);

            // 
            // dcBloodUnitGUID
            // 
            gridTextBoxCol = new DataGridTextBoxColumn();
            gridTextBoxCol.MappingName = TABLE.BloodUnit.BloodUnitGuid;
            gridTextBoxCol.ReadOnly = true;
            gridTextBoxCol.Width = 1;
            gridTableStyle.GridColumnStyles.Add(gridTextBoxCol);

            // 
            // dcInitialABO
            // 
            gridTextBoxCol = new DataGridTextBoxColumn();
            gridTextBoxCol.MappingName = TABLE.ValidAboResult.AboInitial;
            gridTextBoxCol.ReadOnly = true;
            gridTextBoxCol.Width = 0;
            gridTableStyle.GridColumnStyles.Add(gridTextBoxCol);

            // 
            // dcInitialRh
            // 
            gridTextBoxCol = new DataGridTextBoxColumn();
            gridTextBoxCol.MappingName = TABLE.ValidRhResult.RhInitial;
            gridTextBoxCol.ReadOnly = true;
            gridTextBoxCol.Width = 0;
            gridTableStyle.GridColumnStyles.Add(gridTextBoxCol);

            // 
            // dcIndicatorMessage
            // 
            gridTextBoxCol = new DataGridTextBoxColumn();
            gridTextBoxCol.MappingName = INDICATOR_MSG;
            gridTextBoxCol.ReadOnly = true;
            gridTextBoxCol.Width = 0;
            gridTableStyle.GridColumnStyles.Add(gridTextBoxCol);

            dgBloodUnitTests.TableStyles.Add(gridTableStyle);
               
            
        }


        ///<Developers>
        ///	<Developer>Mike Smith</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/13/2002</CreationDate>
        ///
        /// <summary>
        /// Adds all the blood units from the worklist to the unit test grid. 
        /// </summary>
        private void PopulateBloodUnitTests()
        {
	
            foreach(System.Data.DataRow drBloodUnits in worklistBuilder1.SelectedBloodBags.Rows)
            {
				bool performDControl;
				bool performAntiAB;

                BOL.BloodUnit tmpBU = new BOL.BloodUnit(drBloodUnits);

				DataTable dtDailyQCReagentTypes = BOL.ReagentType.GetDailyQCReagentTypes(Common.LogonUser.LogonUserDivisionCode);

				if (dtDailyQCReagentTypes.Select("ReagentTypeId = " + (int)Common.WeakDTestReagents.AntiDControl).Length > 0)
					performDControl = true;
				else
					performDControl = false;

				if (dtDailyQCReagentTypes.Select("ReagentTypeId = " + (int)Common.ReagentType.AntiAB).Length > 0)
					performAntiAB = true;
				else
					performAntiAB = false;

				DataRow drTest = _dtBloodUnitTests.NewRow();

                drTest[ARTIFICIAL.Indicator] = String.Empty;
                drTest[TABLE.BloodUnit.EyeReadableUnitId] = tmpBU.EyeReadableUnitId;
				drTest[TABLE.BloodProduct.ProductCode] = tmpBU.ProductCode;

                drTest[TABLE.ValidAboResult.AntiA] = String.Empty;
                drTest[TABLE.ValidAboResult.AntiB] = String.Empty;
				
				if (this._performAntiD == true)
				{
					drTest[TABLE.ValidRhResult.AntiD] = String.Empty;
					if (performDControl == true)
					{
						drTest[TABLE.ValidRhResult.DControl] = String.Empty;
					}
					drTest[TABLE.ValidRhResult.RhInterp] = String.Empty;
				}
				if (performAntiAB == true)
				{
					 drTest[TABLE.ValidAboResult.AntiAB] = String.Empty;
				}
				
                drTest[TABLE.ValidAboResult.AboInterp] = String.Empty;
                drTest[ARTIFICIAL.AboComment] = String.Empty;
                drTest[ARTIFICIAL.RhComment] = String.Empty;
                drTest[TABLE.BloodUnit.BloodUnitGuid] = tmpBU.BloodUnitGuid;
				
				drTest[TABLE.ValidAboResult.AboInitial] = tmpBU.BloodUnitMedia.AboRh.Abo.ToString();

                //CR3558 (used to be == Positive, but now this way will count Pooled Rh as POS - which is what we want per the BR2.10? )
                drTest[TABLE.ValidRhResult.RhInitial] = (BOL.AboRh.GetRHFromCode(tmpBU.BloodUnitMedia.AboRh.RHFactorCode) != Common.RH.Negative);   
                drTest[INDICATOR_MSG] = String.Empty;
                drTest[TABLE.BloodUnitStatus.BloodUnitStatusGuid] = tmpBU.BloodUnitStatus.BloodUnitStatusGuid;
				drTest[TABLE.BloodUnitStatus.RowVersion] = tmpBU.BloodUnitStatus.RowVersion;

				drTest[TABLE.BloodUnitStatus.QuarantineIndicator] = tmpBU.BloodUnitStatus.QuarantineIndicator;	//CR692
				
				//BR_2.07 -- Find out if RH Confirmation is required - updated check due to CR3558
				drTest[ARTIFICIAL.BR207] = (tmpBU.ProductType.AboRhConfirmationRequired && tmpBU.BloodUnitMedia.AboRh.RH == Common.RH.Negative);

				this._dtBloodUnitTests.Rows.Add(drTest);
            }

        }

        ///<Developers>
        ///	<Developer>Mike Smith</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/13/2002</CreationDate>
        ///
        /// <summary>
        /// Displays the valid values for each column in the legend.
        /// </summary>
        /// <param name="_column"></param>
        /// <param name="_row"></param>
        private void PopulateKeyLegend(int _column, int _row)
        {		
			this.dgBloodUnitTests.Invalidate(true);
			this.dgBloodUnitTests.Update();

            string _validEntries = "";

			switch(_column)
			{
				//All the Anti -- columnns		//1-30-2005 Lohse - Removing P = Positive and N = Negative CR332
				case LEFTMOST_COLUMN: case 3: case 4: case 5:	case 6:
					_validEntries =  "0 = No Agglutination    1 = 1+    2 = 2+    3 = 3+    4 = 4+\n";
					_validEntries += "W = Weak          F = Mixed Field\n";
					_validEntries += "M = Microscopic   X = Not Tested";
					break;

				//The ABO Interp column
				case 7:
					_validEntries = "A = Type A    B = Type B    AB = Type AB    O = Type O    I = Inconclusive";
					break;

				//The Rh Interp column
				case 8:
					if ((bool)this._dtBloodUnitTests.Rows[_row][13])		//BR_2.10  Lohse 1-30-2005
						_validEntries = "P = Positive    N = Negative    I = Inconclusive     X = Not Tested";
					else
						_validEntries = "P = Positive    N = Negative    I = Inconclusive";

					break;

				//The comments columns
				case 9: case RIGHTMOST_COLUMN:
					_validEntries = "Any Text";
					break;

				default:
					_validEntries = "Read-Only";
					break;
			}

            lblValidEntries.Text = _validEntries;
        }

		/// <summary>
		/// Get the exception information
		/// </summary>
		/// <param name="overrideTitle"></param>
		/// <param name="overrideMessage"></param>
		/// <param name="exceptionType"></param>
		/// <returns></returns>
		private BOL.ExceptionReport GetExceptionDetails(string overrideTitle, string overrideMessage, Common.ExceptionType exceptionType)
		{
			
			DlgOverrideComment dlgOverride = new DlgOverrideComment(overrideTitle, overrideMessage , "UT");

            dlgOverride.ShowDialogFromLockedParent(this, false);
			
            if (dlgOverride.DialogResult == DialogResult.OK)
			{
				return(new BOL.ExceptionReport(exceptionType, dlgOverride.Comment, Common.LogonUser.LogonUserName));
			}
			else
			{
				return(null);
			}
		}

		/// <summary>
		/// Processes the current row (input parameter) and determines if it's E,W, or V.  Also will raise any Override Dialogs
		/// </summary>
		/// <param name="row"></param>
		private void UpdateIndicator(int row)
		{
			this._previousRow = row;

			this.dgBloodUnitTests.Invalidate(true);
			this.dgBloodUnitTests.Update();

			string aboIndicator = "E";
			string rhIndicator = "E";		//assume the worst

			string aboMessage = string.Empty;
			string rhMessage = string.Empty;

			DataRow drBloodUnitTest = this._dtBloodUnitTests.Rows[row];

			//If the column is null, that means it's disabled, so we'll set it to a goofy char so translation can function
			string userAntiA		= drBloodUnitTest.IsNull(LEFTMOST_COLUMN) ? "@" : dgBloodUnitTests[row,LEFTMOST_COLUMN].ToString();
			string userAntiB		= drBloodUnitTest.IsNull(3) ? "@" : dgBloodUnitTests[row,3].ToString();
			string userAntiAB		= drBloodUnitTest.IsNull(4) ? "@" : dgBloodUnitTests[row,4].ToString();
			string userAntiD		= drBloodUnitTest.IsNull(5) ? "@" : dgBloodUnitTests[row,5].ToString();
			string userDControl		= drBloodUnitTest.IsNull(6) ? "@" : dgBloodUnitTests[row,6].ToString();
			string userABOInterp	= drBloodUnitTest.IsNull(7) ? "@" : dgBloodUnitTests[row,7].ToString();
			string userRHInterp		= drBloodUnitTest.IsNull(8) ? "@" : dgBloodUnitTests[row,8].ToString();
		
			string initialABO = drBloodUnitTest[12].ToString().Trim();
			string initialRH   = (bool)drBloodUnitTest[13] == true ? "P" : "N";             //Set in PopulateBloodUnitTests()

			bool rhRequired    = (bool)drBloodUnitTest[ARTIFICIAL.BR207];
			bool rhDisabled    = (drBloodUnitTest.IsNull(5) && drBloodUnitTest.IsNull(6));

			if (ChkStrAllowBlank("@", userABOInterp) || (ChkStrAllowBlank("@", userRHInterp) && !rhDisabled))
			{
				drBloodUnitTest[0] = "";
				drBloodUnitTest[TOOLTIP_COLUMN] = "";
				return;
			}

			//Set all the "translated" values
			userAntiA   = this.GetResultTranslation(userAntiA);
			userAntiB   = this.GetResultTranslation(userAntiB);
			userAntiAB  = this.GetResultTranslation(userAntiAB);
			userAntiD   = this.GetResultTranslation(userAntiD);
			userDControl = this.GetResultTranslation(userDControl);
			
			bool BR221Violated = false;
			bool BR217Violated = false;

			//Validate against TT_2.01
			string filterSQL = "AntiA = '" + userAntiA  + "' AND AntiB = '" + userAntiB + "' AND AntiAB = '" + userAntiAB + "' AND AboInitial = '" + initialABO + "'";
			System.Data.DataRow[] drABO = this._dtABOConfirmationLookup.Select(filterSQL);
			if (drABO.Length == 1 && drABO[0]["ABOInterp"].ToString().Trim() == userABOInterp)
			{
				//Valid, Interp and Initial Match! ABOIndicator = "V" 
				aboIndicator = "V";
			}
			else
			{
				if (userABOInterp != string.Empty)
				{ 
					//Error.  Invalid ABO interpretation/results.  ABOIndicator = "E"  -- default
					aboIndicator = "E";
                    aboMessage = Common.StrRes.ValidMsg.UC002.TT201InvalidABO().ResString;

					filterSQL = "AntiA = '" + userAntiA  + "' AND AntiB = '" + userAntiB + "' AND AntiAB = '" + userAntiAB + "'";
					drABO = this._dtABOConfirmationLookup.Select(filterSQL);
 
					if (drABO.Length > 0)		
					{
						string systemABOInterp = drABO[0]["ABOInterp"].ToString().Trim();

						if ((userABOInterp == "I" || userABOInterp == systemABOInterp) && 
							(initialABO != systemABOInterp))
						{
							BR221Violated = (userABOInterp == "I");
							BR217Violated = (userABOInterp != initialABO);

							aboIndicator = "W";
                            aboMessage = Common.StrRes.ValidMsg.UC002.BR17InterpDoesNotMatchLoginABO().ResString;
						}
					}
					else if (UserEnteredJunk(userAntiA, userAntiB , userAntiAB) && userABOInterp == "I")
					{
						BR221Violated = (userABOInterp == "I");
						BR217Violated = (userABOInterp != initialABO);

						aboIndicator = "W";
                        aboMessage = Common.StrRes.ValidMsg.UC002.BR17InterpDoesNotMatchLoginABO().ResString;
					}
				}
			}


			filterSQL = "AntiD = '" + userAntiD  + "' AND RhInterp = '" + userRHInterp + "' AND RHInitial = '" + initialRH + "'";
			if (userDControl != "~") filterSQL = filterSQL + " AND DControl = '" + userDControl + "'";

			//Validate against TT_2.02
			System.Data.DataRow[] drRH = _dtRHConfirmationLookup.Select(filterSQL);
			if (drRH.Length > 0 || (userAntiD == "X" && userDControl == "X" && userRHInterp == "X" && !rhRequired) ||
				(rhDisabled && !rhRequired))	// ||	(userAntiD == "~" && userDControl == "~" && userRHInterp == string.Empty && !rhRequired)) - removing 12/1/05 - NO blanks allowed
			{
				//Valid.  Interp and Initial Match!  RHIndicator = "V"
				rhIndicator = "V";
			}
			else
			{
				if (userRHInterp != string.Empty)
				{
					if (
						(ChkStr("X", userAntiD) && ChkStr("P", userDControl) && ChkStr("PNI", userRHInterp) && ChkStr("P", initialRH)) ||
						(ChkStr("PN", userAntiD) && ChkStr("P", userDControl) && ChkStr("PNI", userRHInterp) && ChkStr("P", initialRH)) ||
						(ChkStr("PN", userAntiD) && ChkStr("P", userDControl) && ChkStr("PNI", userRHInterp) && ChkStr("N", initialRH))
						)
					{
						rhIndicator = "W";
                        rhMessage = Common.StrRes.ValidMsg.UC002.TT202DControlPositive().ResString;

						BR221Violated = true;
					}

					if (
						(ChkStr("N", userAntiD) && ChkStr("~N", userDControl) && ChkStr("N", userRHInterp) && ChkStr("P", initialRH))						)
					{
						rhIndicator = "W";
                        rhMessage = Common.StrRes.ValidMsg.UC002.TT202DControlPositive().ResString;

						BR217Violated = true;
					}

					if (
						(ChkStr("P", userAntiD) && ChkStr("N", userDControl) && ChkStr("PI", userRHInterp) && ChkStr("N", initialRH)) ||
						(ChkStr("P", userAntiD) && ChkStr("~X", userDControl) && ChkStr("PN", userRHInterp) && ChkStr("N", initialRH)) ||
						(ChkStr("N", userAntiD) && ChkStr("~X", userDControl) && ChkStr("PNI", userRHInterp) && ChkStr("P", initialRH)) ||
						(ChkStr("X", userAntiD) && ChkStr("P", userDControl) && ChkStr("PNI", userRHInterp) && ChkStr("N", initialRH))
						)
					{
						rhIndicator = "W";
                        rhMessage = Common.StrRes.ValidMsg.UC002.TT202ResultsDoNotMatchLoginRH().ResString;

						BR217Violated = true;
					}

					//If it hasn't been caught by now, it's an error
					if (rhIndicator == "E")	
					{
						if (
							(ChkStr("X", userAntiD) && ChkStr("N", userDControl) && ChkStr("I", userRHInterp) && ChkStr("PN", initialRH)) ||
							(ChkStr("X", userAntiD) && ChkStr("~X", userDControl) && ChkStr("I", userRHInterp) && ChkStr("P", initialRH)) ||
							(ChkStr("X", userAntiD) && ChkStr("~X", userDControl) && ChkStr("P", userRHInterp) && ChkStr("PN", initialRH)) ||
							(ChkStr("X", userAntiD) && ChkStr("N", userDControl) && ChkStr("PN", userRHInterp) && ChkStr("P", initialRH))
							)
						{
                            rhMessage = Common.StrRes.ValidMsg.UC002.TT202InvalidRhTyping().ResString;
						}				

						if (
							(ChkStr("X", userAntiD) && ChkStr("N", userDControl) && ChkStr("PN", userRHInterp) && ChkStr("N", initialRH)) ||
							(ChkStr("X", userAntiD) && ChkStr("X", userDControl) && ChkStr("XN", userRHInterp) && ChkStr("PN", initialRH)) ||
							(ChkStr("X", userAntiD) && ChkStr("X", userDControl) && ChkStr("I", userRHInterp) && ChkStr("N", initialRH)) ||
							(ChkStr("X", userAntiD) && ChkStr("~", userDControl) && ChkStr("N", userRHInterp) && ChkStr("P", initialRH)) ||
							(ChkStr("X", userAntiD) && ChkStr("~", userDControl) && ChkStr("XIN", userRHInterp) && ChkStr("N", initialRH)) ||
							(ChkStr("~", userAntiD) && ChkStr("~", userDControl) && ChkStr("~X", userRHInterp) && ChkStr("N", initialRH)) 
							)
						{
                            rhMessage = Common.StrRes.ValidMsg.UC002.TT202RhTypingReqdNegUnit().ResString;
						}				

						if (
							(ChkStr("N", userAntiD) && ChkStr("~N", userDControl) && ChkStr("PI", userRHInterp) && ChkStr("PN", initialRH)) ||
							(ChkStr("P", userAntiD) && ChkStr("~X", userDControl) && ChkStr("I", userRHInterp) && ChkStr("PN", initialRH)) ||
							(ChkStr("P", userAntiD) && ChkStr("N", userDControl) && ChkStr("N", userRHInterp) && ChkStr("PN", initialRH)) ||
							(ChkStr("P", userAntiD) && ChkStr("N", userDControl) && ChkStr("I", userRHInterp) && ChkStr("P", initialRH)) ||
							(ChkStr("P", userAntiD) && ChkStr("~X", userDControl) && ChkStr("N", userRHInterp) && ChkStr("P", initialRH)) ||
							(ChkStr("N", userAntiD) && ChkStr("X", userDControl) && ChkStr("PI", userRHInterp) && ChkStr("N", initialRH))
							)
						{
                            rhMessage = Common.StrRes.ValidMsg.UC002.TT202RhInterpDoesNotMatchEntries().ResString;
						}				

					}
				}
			}

			//Raise BR_2.17 exception handling if needed
			if (BR217Violated)
			{
				if (drBloodUnitTest.IsNull(ARTIFICIAL.BR217))
				{
                    BOL.ExceptionReport mismatchException = this.GetExceptionDetails("ABO/Rh Mismatch", Common.StrRes.ValidMsg.UC002.BR17ExceptionTitle().ResString, Common.ExceptionType.ABORh_confirmation_does_not_match_login_ABORh);
					if (mismatchException == null)
					{
						drBloodUnitTest[ARTIFICIAL.BR217] = DBNull.Value;
						drBloodUnitTest[this._currentColumn] = this._prevCellValue;
						return;

					}
					else
					{
						mismatchException.GenerateExceptionUnitAboRh((Guid)drBloodUnitTest[TABLE.BloodUnit.BloodUnitGuid], userABOInterp, userRHInterp, this._testUser);
						drBloodUnitTest[ARTIFICIAL.BR217] = mismatchException;
					}
				}
			}
			else		//Reset the column if everything is OK
			{
				drBloodUnitTest[ARTIFICIAL.BR217] = DBNull.Value;
			}

			//Raise BR_2.21 exception handling if needed
			if (BR221Violated)
			{
				if (drBloodUnitTest.IsNull(ARTIFICIAL.BR221))
				{
                    BOL.ExceptionReport rhInconclusiveException = this.GetExceptionDetails("Inconclusive ABO/Rh", Common.StrRes.ValidMsg.UC002.BR21ExceptionTitle().ResString, Common.ExceptionType.ABORh_confirmation_inconclusive_unit_quarantined);
					if (rhInconclusiveException == null)
					{
						drBloodUnitTest[ARTIFICIAL.BR221] = DBNull.Value;
						drBloodUnitTest[this._currentColumn] = this._prevCellValue;
						return;
					}
					else
					{
						rhInconclusiveException.GenerateExceptionUnitAboRh((Guid)drBloodUnitTest[TABLE.BloodUnit.BloodUnitGuid], userABOInterp, userRHInterp, this._testUser);
						drBloodUnitTest[ARTIFICIAL.BR221] = rhInconclusiveException;
					}
				}
			}
			else
			{
				drBloodUnitTest[ARTIFICIAL.BR221] = DBNull.Value;
			}

            if (aboIndicator == "E" && aboMessage == string.Empty) aboMessage = Common.StrRes.Tooltips.UC002.IncompleteABOResults().ResString;
            if (rhIndicator == "E" && rhMessage == string.Empty) rhMessage = Common.StrRes.Tooltips.UC002.IncompleteRhResults().ResString;


			//Set the ERROR ICON and TOOLTIP
			if (aboIndicator == "V" && rhIndicator == "V")
				drBloodUnitTest[0] = "V";
			else
			{
				if ((ChkStr("E", aboIndicator) || ChkStr("E", rhIndicator)))
				{
					drBloodUnitTest[0] = "E";
				}
				else
					drBloodUnitTest[0] = "W";
			}


			//Format the tooltip message properly
			if (aboMessage == string.Empty)
			{
				drBloodUnitTest[TOOLTIP_COLUMN] = rhMessage;
			}
			else
			{
				drBloodUnitTest[TOOLTIP_COLUMN] = aboMessage;

				if (rhMessage != string.Empty)
				{
					drBloodUnitTest[TOOLTIP_COLUMN] = drBloodUnitTest[TOOLTIP_COLUMN].ToString() + "\n" + rhMessage;
				}
			}
		}

		//Saves on typing
		private bool ChkStr (string possibleValues, string inputString)
		{
			if (inputString == "") return(false);

			return (possibleValues.IndexOf(inputString) > -1);
		}

		private bool ChkStrAllowBlank (string possibleValues, string inputString)
		{
			//Will return true if inputString is blank
			return (possibleValues.IndexOf(inputString) > -1);
		}

		/// <summary>
		/// This is to covered the bottom half of the "Warning" section of TT2.01
		/// </summary>
		/// <param name="antiA"></param>
		/// <param name="antiB"></param>
		/// <param name="antiAB"></param>
		/// <returns></returns>
		private bool UserEnteredJunk(string antiA, string antiB, string antiAB)
		{

			if (
				(antiA == "N" && antiB == "N" && antiAB == "P") ||
				(antiA == "P" && antiB == "P" && antiAB == "N") ||
				(antiA == "P" && antiB == "N" && antiAB == "N") ||
				(antiA == "N" && antiB == "P" && antiAB == "N") )
			{
				return(true);
			}
			else
				return(false);
		}

        ///<Developers>
        ///	<Developer>Mike Smith</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/13/2002</CreationDate>
        ///
        /// <summary>
        /// Displays a confirmation message to the user and saves the test results for all entries
        /// that are valid or contain warnings.
        /// </summary>
        private void SaveBloodUnitTests()
        {
            Common.WorkloadProcessID workloadProcessId = Common.WorkloadProcessID.Unknown;   //CR3558 no need for array, will only be a single ID per Save
			string message = "";
			
			int invalidUnits = 0;

			//Implementing BR_2.44
			System.Data.DataTable dt = null;
			foreach (System.Data.DataRow drTest in this._dtBloodUnitTests.Rows)
			{
                //CR3558 - fixing workload
				if (drTest[0].ToString() != "W" && drTest[0].ToString() != "V")
				{
					invalidUnits++;
				}
			}



			//CR 2707 - modified message that is being displayed upon save. List of blood units is no longer displayed.
			if (invalidUnits > 0)
			{
				message += Common.StrRes.ConfMsg.UC002.IncompleteTestsNotSaved().ResString;
				message += "\n";
			}
            
			message += Common.StrRes.ConfMsg.Common.OKToSave("ABO/Rh blood unit test").ResString;
			
			DialogResult continueWithSave = GuiMessenger.ShowMessageBox(this,Common.StrRes.ConfMsg.Common.Dynamic(message));

            if (continueWithSave == DialogResult.Yes)
            {
                try
                {
                    //CR 1441  -- Workload should only be applied to "saveable" test results -CR3558 updated -  BOL/DAL will omit non W/V interps
                    if (this._performAntiD == false)
                    {
                        dt = BOL.WorkloadEvent.GetWorkloadProcessGuidByProcessID((int)Common.WorkloadProcessID.ABOForwardTypingUnit);
                        workloadProcessId = Common.WorkloadProcessID.ABOForwardTypingUnit;
                    }
                    else
                    {
                        dt = BOL.WorkloadEvent.GetWorkloadProcessGuidByProcessID((int)Common.WorkloadProcessID.ABORhForwardTypingUnit);
                        workloadProcessId = Common.WorkloadProcessID.ABORhForwardTypingUnit;
                    }

                    BOL.BloodUnitTest.SaveAboRHBloodUnitTests(_dtBloodUnitTests, this._testDateTime, this._testUser, this._automatedInstrName, this._automatedInstrQCd, this._rackGuid, Common.UpdateFunction.UC002FrmAboRHConfirmProcess, workloadProcessId, this._rackNotQcComment);
                    //Setting the dirty variable of the base form to false so
					//that when the page is closing that it will not prompt the user
					//with the unsaved message prompt.
					//CR 1441  
					if(dt!= null)
					{
						if(dt.Rows.Count == 0)
						{
							GuiMessenger.ShowMessageBox(Common.StrRes.InfoMsg.UC015.BR1502_NoWorkloadDefined());
						}
					}
					this.IsDirty=false;
					this.Close();
                }
				catch(Common.RowVersionException)
				{
					this.btnOK.RowVersionException = true;		//CR2077
					GuiMessenger.ShowMessageBox(Common.StrRes.SysErrMsg.Common.DataWasNotSavedBecauseOfRowversionViolation());
				}
			}				
        }

		#endregion
		
		#region Events

        ///<Developers>
        ///	<Developer>Mike Smith</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/13/2002</CreationDate>
        ///
        /// <summary>
        /// This code traps each key stroke to allow movement to the next cell for the columns that only 
        /// require a single valid key stroke.  TT_2.01, BR_2.35, TT_2.02,BR_2.10, BR_2.05, BR_2.06, BR_2.24
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void gridKeyTrapTextBoxCol_TextBoxKeyPress(object sender, System.Windows.Forms.KeyPressEventArgs e)
        {
			try
			{
				this.dgBloodUnitTests.Invalidate(true);
				this.dgBloodUnitTests.Update();

				//Need to set this manually, because DataGrid keypress doesn't count for FormLock
				this.ResetLockTimer(this, System.EventArgs.Empty);

				int rowNum = dgBloodUnitTests.CurrentCell.RowNumber;
				int colNum = dgBloodUnitTests.CurrentCell.ColumnNumber;

				string cellText, possibleText, keyChar, validKeys =  String.Empty;

				TextBox c = (TextBox) sender;

				cellText = c.Text;

				this._prevCellValue = cellText;
				this._currentColumn = colNum;

				// Ignore if column does not require validation treatment
				if (colNum > _EndEditColumn && colNum <= RIGHTMOST_COLUMN)
				{
					e.Handled = true;
					this.dgBloodUnitTests.Invalidate(true);
					this.dgBloodUnitTests.Update();
					return;
				}

				// Id, indicator, and hidden columns are not editable
				if (colNum < LEFTMOST_COLUMN || colNum > RIGHTMOST_COLUMN)
				{
					e.Handled = true;
					this.dgBloodUnitTests.Invalidate(true);
					this.dgBloodUnitTests.Update();
					return;
				}
		
				// Let backspace and enter keys bypass
				if (!char.IsLetterOrDigit(e.KeyChar))
				{
					if (e.KeyChar != Convert.ToChar(Keys.Enter))
					{
						e.Handled = !char.IsControl(e.KeyChar);
						this.dgBloodUnitTests.Invalidate(true);
						this.dgBloodUnitTests.Update();
						return;
					}
					else
					{
						if (rowNum == this._dtBloodUnitTests.Rows.Count - 1)
						{
							if (colNum < 11 && colNum > 1)
							{
								this.dgBloodUnitTests.CurrentCell = new DataGridCell(0, colNum+1);
							}
							else if (colNum == 10)
							{
								this.dgBloodUnitTests.CurrentCell = new DataGridCell(0, 0);
							}
						}
						else
						{
							this.dgBloodUnitTests.CurrentCell = new DataGridCell(rowNum, colNum+1);
						}
						this.dgBloodUnitTests.Invalidate(true);
						this.dgBloodUnitTests.Update();
						return;
					}
				}
		
				// Turn single character to upper
				keyChar = e.KeyChar.ToString().ToUpper()[0].ToString();

				// Obtain what the string will look like after key is pressed
				if (cellText.Length > 0)
				{
					possibleText = cellText.Substring(0, c.SelectionStart) + keyChar + cellText.Substring(c.SelectionStart + c.SelectionLength, cellText.Length - (c.SelectionLength + c.SelectionStart));
				}
				else
				{
					possibleText = keyChar;
				}

				// Test what type of column it is and then apply the regular expression
				if (colNum == 7)  // COLTYPE_ABO
				{
					validKeys = @"(\A[ABOI]\z)|(\AAB\z)";
				}
				else if (colNum == 8)  // COLTYPE_RH		//1-30-2005 - Lohse, Adding X BR_2.10
				{
					if ((bool)this._dtBloodUnitTests.Rows[rowNum][13])
						validKeys = @"(\A[PNIX]\z)";
					else
						validKeys = @"(\A[PNI]\z)";
				}
				else if (colNum == LEFTMOST_COLUMN || colNum == 3 ||    // COLTYPE_ANTIA,  COLTYPE_ANTIB
					colNum == 4 || colNum == 5 ||			// COLTYPE_ANTIAB, COLTYPE_ANTID
					colNum == 6)							// COLTYPE_DCONTROL
				{
					//5/15/2004 Lohse - According to TT_2.01 - Removing Hemolysis and Roulouex  (H,R)
					//1-30-2005 Lohse - CR332 - Removing P,N
					validKeys = @"(\A[WFM01234X]\z)";
				}

				if (Regex.IsMatch(possibleText, validKeys))
				{
					//changing values in the datable causes TextChanged event to be fired
					//setting a flag to prevent it from happening
					this._keyPressEventHandled = true;
					c.Text = possibleText;
					c.SelectionStart = c.SelectionLength + 2;
					//
					DataTable dt = ((DataView)this.dgBloodUnitTests.DataSource).Table;
					dt.Rows[rowNum][colNum] = c.Text;

					if (colNum != 7)  
					{
						if (rowNum == this._dtBloodUnitTests.Rows.Count - 1)
						{
							if (colNum < 11 && colNum > 1)
							{
								this.dgBloodUnitTests.CurrentCell = new DataGridCell(rowNum, colNum+1);
							}
							else if (colNum == 10)
							{
								this.dgBloodUnitTests.CurrentCell = new DataGridCell(0, 0);
							}
						}
						else
						{
							this.dgBloodUnitTests.CurrentCell = new DataGridCell(rowNum, colNum + 1);
						}
					}
					this._keyPressEventHandled = false;
				}
				else
				{
					Common.Utility.MessageBeep();
					GuiMessenger.ShowMessageBox(this, Common.StrRes.SysErrMsg.Common.InvalidKeyPressed());
				}

				this.dgBloodUnitTests.Invalidate(true);
				this.dgBloodUnitTests.Update();

				e.Handled = true;
				return;			
			}
			catch(ObjectDisposedException) {} //swallow the locking pill
        }

		/// <summary>
		/// This method handles entering invalid entries into the grid (CR 2792)
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void gridKeyTrapTextBoxCol_TextChanged(object sender, EventArgs e)
		{
			if (this._keyPressEventHandled || this.dgBloodUnitTests.EnterKeyPressed == true)
			{
				return;
			}
			else
			{
				int row = this.dgBloodUnitTests.CurrentCell.RowNumber;
				int column = this.dgBloodUnitTests.CurrentCell.ColumnNumber;
				DataTable dt = ((DataView)this.dgBloodUnitTests.DataSource).Table;
			
				//forcing the validation of the grid
				if (sender is System.Windows.Forms.DataGridTextBox)
				{
					//if the grid cell has anything in it run the regex match
					if (((DataGridTextBox)sender).Text != "" && ((DataGridTextBox)sender).Text != "(null)")
					{
						string regPattern = "";

						// Test what type of column it is and then apply the regular expression
						if (column == 7)  // COLTYPE_ABO
						{
							regPattern = @"(\A[ABOI]\z)|(\AAB\z)";
						}
						else if (column == 8)  // COLTYPE_RH
						{
							if ((bool)this._dtBloodUnitTests.Rows[row][13])
								regPattern = @"(\A[PNIX]\z)";
							else
								regPattern = @"(\A[PNI]\z)";
						}
						else if (column == LEFTMOST_COLUMN || column == 3 ||    // COLTYPE_ANTIA,  COLTYPE_ANTIB
							column == 4 || column == 5 ||			// COLTYPE_ANTIAB, COLTYPE_ANTID
							column == 6)							// COLTYPE_DCONTROL
						{
							regPattern = @"(\A[WFM01234X]\z)";
						}

						if (Regex.IsMatch(((DataGridTextBox)sender).Text, regPattern))
						{
							//changing values in the datable causes TextChanged event to be fired
							//setting a flag to prevent it from happening
							this._keyPressEventHandled = true;
							dt.Rows[row][column] = ((DataGridTextBox)sender).Text;
							
							UpdateIndicator(row);
							this.dgBloodUnitTests.Refresh();
							this.CheckValid(this, System.EventArgs.Empty);

							this._keyPressEventHandled = false;
						}
						else
						{
							//changing values in the datable causes TextChanged event to be fired
							//setting a flag to prevent it from happening
							this._keyPressEventHandled = true;
							((DataGridTextBox)sender).Text = dt.Rows[row][column].ToString();
							this._keyPressEventHandled = false;
							//BR_2.28
							GuiMessenger.ShowMessageBox(Common.StrRes.ValidMsg.Common.InvalidKeyPressed());
						}
					}
						//user hit Del key and the grid cell is blank - refresh grid, do not run regex match
					else
					{
						//changing values in the datable causes TextChanged event to be fired
						//setting a flag to prevent it from happening
						this._keyPressEventHandled = true;
						dt.Rows[row][column] = "";

						UpdateIndicator(row);
						this.dgBloodUnitTests.Refresh();
						this.CheckValid(this, System.EventArgs.Empty);
						
						this._keyPressEventHandled = false;
					}
				}
			}
		}

		private void dgBloodUnitTests_SizeChanged(object sender, System.EventArgs e)
		{
			try
			{
				if (this.dgBloodUnitTests.TableStyles.Count > 0)
				{
					DataGridTableStyle tblStyle = this.dgBloodUnitTests.TableStyles[0];
					
					tblStyle.GridColumnStyles[ARTIFICIAL.AboComment].Width = (int)((this.dgBloodUnitTests.Width -
						(tblStyle.GridColumnStyles[ARTIFICIAL.Indicator].Width +
						tblStyle.GridColumnStyles[TABLE.BloodUnit.EyeReadableUnitId].Width + 
						tblStyle.GridColumnStyles[TABLE.ValidAboResult.AntiA].Width +
						tblStyle.GridColumnStyles[TABLE.ValidAboResult.AntiB].Width +
						tblStyle.GridColumnStyles[TABLE.ValidAboResult.AntiAB].Width +
						tblStyle.GridColumnStyles[TABLE.ValidRhResult.AntiD].Width +
						tblStyle.GridColumnStyles[TABLE.ValidRhResult.DControl].Width +
						tblStyle.GridColumnStyles[TABLE.ValidAboResult.AboInterp].Width +
						tblStyle.GridColumnStyles[TABLE.ValidRhResult.RhInterp].Width + 7)) / 2);


					tblStyle.GridColumnStyles[ARTIFICIAL.RhComment].Width = tblStyle.GridColumnStyles[ARTIFICIAL.AboComment].Width;

					this.cmbABOComment.Width = tblStyle.GridColumnStyles[ARTIFICIAL.AboComment].Width;
					this.cmbRhComment.Width = tblStyle.GridColumnStyles[ARTIFICIAL.RhComment].Width;
				}
			}
			catch(ObjectDisposedException){/*swallow the ObjectDisposedException ocurring on a lock expiration*/}
		}


        ///<Developers>
        ///	<Developer>Mike Smith</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/13/2002</CreationDate>
        ///
        /// <summary>
        /// Updates the indicator flag after each cell is edited.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void dgBloodUnitTests_CurrentCellChanged(object sender, System.EventArgs e)
        {
			this.dgBloodUnitTests.Invalidate(true);
			this.dgBloodUnitTests.Update();

			int row = dgBloodUnitTests.CurrentCell.RowNumber;
			int col = dgBloodUnitTests.CurrentCell.ColumnNumber;

			if (this._previousRow != -1 && this._previousRow != row)
			{
				this.UpdateIndicator(this._previousRow);
			}

			UpdateIndicator(row);

			this.CheckValid(this, System.EventArgs.Empty);

			this.dgBloodUnitTests.Invalidate(true);
			this.dgBloodUnitTests.Update();

			this.PopulateKeyLegend(col, row);

			if (this._manualAdvance) return;

			this.cmbRhComment.Visible = false;
			this.cmbABOComment.Visible = false;

			if (this._byPassColumn)
			{
				this._byPassColumn = false;
				return;
			}

			if (this._dtBloodUnitTests.Rows[row].IsNull(col))
				this.dgBloodUnitTests.CurrentCell = new DataGridCell(row, col+1);


			if(dgBloodUnitTests.CurrentCell.ColumnNumber < LEFTMOST_COLUMN)
			{
				this._byPassColumn = true;
				dgBloodUnitTests.CurrentCell = new DataGridCell(dgBloodUnitTests.CurrentCell.RowNumber, LEFTMOST_COLUMN); 
			}

			if(dgBloodUnitTests.CurrentCell.ColumnNumber > RIGHTMOST_COLUMN)
			{
				this._byPassColumn = true;
				dgBloodUnitTests.CurrentCell = new DataGridCell(dgBloodUnitTests.CurrentCell.RowNumber, RIGHTMOST_COLUMN); 
			}

			PopulateKeyLegend(dgBloodUnitTests.CurrentCell.ColumnNumber, dgBloodUnitTests.CurrentCell.RowNumber);
			
			if (dgBloodUnitTests.CurrentCell.ColumnNumber == 9)
			{
				cmbABOComment.Text = ((DataView)this.dgBloodUnitTests.DataSource).Table.Rows[this.dgBloodUnitTests.CurrentRowIndex][ARTIFICIAL.AboComment].ToString();
				//
				cmbABOComment.Visible = true;
				this.cmbABOComment.Show();
				cmbABOComment.Focus();
			}
			
			if (dgBloodUnitTests.CurrentCell.ColumnNumber == RIGHTMOST_COLUMN)
			{
				cmbRhComment.Text = ((DataView)this.dgBloodUnitTests.DataSource).Table.Rows[this.dgBloodUnitTests.CurrentRowIndex][ARTIFICIAL.RhComment].ToString();
				//
				cmbRhComment.Visible = true;
				this.cmbRhComment.Show();
				cmbRhComment.Focus();
			}

			this.IsDirty = true;
		}

        ///<Developers>
        ///	<Developer>Mike Smith</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/13/2002</CreationDate>
        ///
        /// <summary>
        /// If the worklist builder is the visible, fill the datagrid with the selected blood
        /// units that need to be tested.  If the datagrid is visible, confirm the blood unit
        /// tests and save them if they are confirmed.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnOK_Click(object sender, System.EventArgs e)
        {
            if (pnlWorklist.Visible)		//User clicked OK after selecting units for worklist
            {
				System.Collections.ArrayList selectedUnits = this.worklistBuilder1.SelectedBloodUnits;

				foreach (BOL.BloodUnit unit in selectedUnits)
				{
                    this.LockController.RecordGuids.Add(unit.BloodUnitStatus.BloodUnitStatusGuid);
				}
				
                //Now that we have added all the BUS guids to the LockController, flip the switch
				if (this.SetLocks())
				{
					//Flip flop those panels
					pnlConfirm.Visible = true;
					pnlWorklist.Visible = false;
					this.SetThisHelpFile(this, Common.StrRes.HelpFiles.UC002.FrmAboRHConfirmProcess_Worksheet());

					InitializeGrid();
					this.btnOK.EnabledButton = false;
                    this.btnOK.SetToolTip(Common.StrRes.Tooltips.UC002.NeedOneValidTestToSave().ResString);
				}
				else
				{
					//Locks weren't set, so leave the user on this screen to select again
				}
            }
            else
            {
				//CR 2792 Additional check to see if data on the grid are valid
				this.CheckValid(this, System.EventArgs.Empty);
				if (this.btnOK.EnabledButton == false)
					return;

				SaveBloodUnitTests();
            }
        }

        ///<Developers>
        ///	<Developer>Mike Smith</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/13/2002</CreationDate>
        ///
        /// <summary>
        /// Worklist builder control validity event handler, enables OK button when at least one unit is selected
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void CheckValid(object sender, System.EventArgs e)
        {
			if (this.pnlWorklist.Visible)
			{
				if (this.worklistBuilder1.SelectedItems > 0)
				{
					this.btnOK.EnabledButton = true;
					this.btnOK.SetToolTip(Common.StrRes.Tooltips.Common.OKToContinue().ResString);
				}
				else
				{
					this.btnOK.EnabledButton = false;
					this.btnOK.SetToolTip(Common.StrRes.Tooltips.UC002.SelectUnit().ResString);
				}
			}
			else
			{

				bool hasValidData = false;

				foreach (System.Data.DataRow drTest in this._dtBloodUnitTests.Rows)
				{
					if (drTest[0].ToString() == "W" ||
						drTest[0].ToString() == "V")
					{
						hasValidData = true;
						break;
					}
				}

				if (hasValidData)
				{
					this.btnOK.SetToolTip(Common.StrRes.Tooltips.Common.OKToSave().ResString);
					this.btnOK.EnabledButton = true;
				}
				else
				{
					this.btnOK.EnabledButton = false;
                    this.btnOK.SetToolTip(Common.StrRes.Tooltips.UC002.NeedOneValidTestToSave().ResString);
				}

			}
        }

        ///<Developers>
        ///	<Developer>Mike Smith</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>12/13/2002</CreationDate>
        ///
        /// <summary>
        /// btnCancel_Click Closes the form.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btnCancel_Click(object sender, System.EventArgs e)
        {
            this.Close();
        }

        ///<Developers>
        ///	<Developer>Mike Smith</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>2/10/2003</CreationDate>
        ///
        /// <summary>
        ///  Searches for new limited blood units to add to the worklist builder.
        /// </summary>
        /// <param name="sender">object</param>
        /// <param name="e">System.EventArgs</param>
        private void worklistBuilder1_SearchClick(object sender, System.EventArgs e)
        {
            using ( new WaitCursor() ) 
            {
				BOL.BloodUnit addingUnit = null;

				//Get the right list of "available units" 
				System.Data.DataTable dtLimitedUnits = BOL.BloodUnit.GetLimitedBloodUnitsByDivision(Common.LogonUser.LogonUserDivisionCode, this._performAntiD);						

				DlgSelectBloodUnits unitSelector = new DlgSelectBloodUnits(dtLimitedUnits, this.worklistBuilder1.bloodUnitSelector1.EnteredEyeReadableUnitId, this.worklistBuilder1.bloodUnitSelector1.EnteredUnitProductCode);
                unitSelector.MultiSelect = true;
                unitSelector.TitleBarText = "VBECS - Select Units for ABO/Rh confirmation";
                unitSelector.LogoBarText = "Select Units for ABO/Rh confirmation";
                unitSelector.ShowDialog(this);
				
                if (unitSelector.DialogResult == DialogResult.OK)
                {
					this.worklistBuilder1.ResetSelectorFieldsAndErrors();

                    foreach (DataRow dr in unitSelector.SelectedUnits.Rows)
                    {
						try
						{
							addingUnit = new BOL.BloodUnit((System.Guid)dr[TABLE.BloodUnit.BloodUnitGuid]);
							this.worklistBuilder1.PopulateGrid(addingUnit);
						}
						catch(Exception err)
						{
							GuiMessenger.ShowMessageBox(this, Common.StrRes.SysErrMsg.Common.Dynamic(err.Message));
						}
                    }
                }
            }	
        }

		/// <summary>
		/// Gets the BloodUnit for ABO/Rh Confirmation, based on BloodUnitSelector entry
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void worklistBuilder1_SelectBtnClick(object sender, System.EventArgs e)
		{
            System.Data.DataTable _eligibleUnits = BOL.BloodUnit.GetLimitedBloodUnit(this.worklistBuilder1.bloodUnitSelector1.BloodUnit.BloodUnitGuid);

			try
			{
				//If the unit isn't valid, don't let them select it
                if (_eligibleUnits.Rows.Count == 0)
				{
					//We needs to throw a exception to trip up the BloodUnitSelector - now more generic since BRs were cleaned up - Defect 251426
                    throw new BOL.BusinessObjectException(Common.StrRes.ValidMsg.UC002.UnitNotEligibleForConfirmation(this.worklistBuilder1.bloodUnitSelector1.BloodUnit.EyeReadableUnitId, this.worklistBuilder1.bloodUnitSelector1.BloodUnit.ProductCode).ResString);
				}
				else
				{
                    if (BOL.AboRh.GetRHFromCode(_eligibleUnits.Rows[0][TABLE.RhFactor.RhFactorCode].ToString()) == Common.RH.Negative && !this._performAntiD)
					{
                        throw new BOL.BusinessObjectException(Common.StrRes.ValidMsg.UC002.BR07RhNegUnitsNeedAntiDTest().ResString);
					}
					//the base handler should add this unit
				}
			}
			catch(Exception err)
			{
				GuiMessenger.ShowMessageBox(Common.StrRes.InfoMsg.Common.Dynamic(err.Message));
				throw;
			}
		}


        private void FrmAboRHConfirmProcess_Load(object sender, System.EventArgs e)
        {
            this.worklistBuilder1.Select();
            this.worklistBuilder1.Focus();
        }

		#endregion

        private void cmbABOComment_KeyDown(object sender, KeyEventArgs e)
        {            
			if (e.KeyCode == Keys.Enter)
            {
                dgBloodUnitTests.CurrentCell = new DataGridCell(dgBloodUnitTests.CurrentCell.RowNumber, dgBloodUnitTests.CurrentCell.ColumnNumber + 1);

                cmbABOComment.Visible = false;
                e.Handled = true;
            }
        }

		private void cmbABOComment_VisibleChanged(object sender, System.EventArgs e)
		{
			if (!cmbABOComment.Visible) return;

			DataTable dt = ((DataView)this.dgBloodUnitTests.DataSource).Table;

			if (!dt.Rows[dgBloodUnitTests.CurrentCell.RowNumber].IsNull(ARTIFICIAL.AboComment))
			{
				cmbABOComment.Text = dt.Rows[dgBloodUnitTests.CurrentCell.RowNumber][ARTIFICIAL.AboComment].ToString();
			}
		}

        private void cmbABOComment_TextChanged(object sender, System.EventArgs e)
        {
			this.ResetLockTimer(this, System.EventArgs.Empty);

            DataTable dt = ((DataView)this.dgBloodUnitTests.DataSource).Table;

			dt.Rows[dgBloodUnitTests.CurrentCell.RowNumber][ARTIFICIAL.AboComment] = cmbABOComment.Text;
        }

		private void cmbRhComment_KeyDown(object sender, KeyEventArgs e)
		{

			if (e.KeyCode == Keys.Enter)
			{
				cmbRhComment.Visible = false;
				e.Handled = true;
			}
		}

        private void cmbRhComment_TextChanged(object sender, System.EventArgs e)
        {
			this.ResetLockTimer(this, System.EventArgs.Empty);

            DataTable dt = ((DataView)this.dgBloodUnitTests.DataSource).Table;
			
			dt.Rows[dgBloodUnitTests.CurrentCell.RowNumber][ARTIFICIAL.RhComment] = cmbRhComment.Text;
        }

		private void cmbRhComment_VisibleChanged(object sender, System.EventArgs e)
		{
			if (!cmbRhComment.Visible) return;

			DataTable dt = ((DataView)this.dgBloodUnitTests.DataSource).Table;

			if (!dt.Rows[dgBloodUnitTests.CurrentCell.RowNumber].IsNull(ARTIFICIAL.RhComment))
			{
				cmbRhComment.Text = dt.Rows[dgBloodUnitTests.CurrentCell.RowNumber][ARTIFICIAL.RhComment].ToString();
			}
		}

		/// <summary>
		/// Enables/Disables OK button based on box checking
		/// </summary>
		/// <param name="sender">BloodUnit being added/removed</param>
		/// <param name="e">System.Windows.Forms.ItemCheckEventArgs</param>
        private void worklistBuilder1_ItemCheck(object sender, System.EventArgs e)
        {
			this.CheckValid(this, System.EventArgs.Empty);
        }

		private void pnlConfirm_VisibleChanged(object sender, System.EventArgs e)
		{
			if (this.pnlConfirm.Visible)
			{
				this.btnOK.Image = controls.VbecsImage.Save.Image;
			}
			else
			{
				this.btnOK.Image = controls.VbecsImage.Transparent.Image;
			}
		}

    }
}
